/**
* @par Copyright (C): 2010-2019, Shenzhen Yahboom Tech
* @file         RGB.ino
* @author       chengengyue
* @version      V1.0
* @date         2019.10.08
* @brief        点亮RGB灯
* @details
* @par History  见如下说明
*/
// 导入库文件
#include <Adafruit_NeoPixel.h>

#define BUZZER 10    // 定义蜂鸣器引脚
#define KEY_PIN 8    // 定义功能按键引脚
#define LED_PIN 5    // 定义LED灯（D9）引脚
#define RGB_PIN 9    // 定义RGB灯的引脚
#define MAX_RGB 4    // 小车一共有4个RGB灯

// 初始化WS2812编程灯
Adafruit_NeoPixel strip = Adafruit_NeoPixel(MAX_RGB, RGB_PIN, NEO_RGB + NEO_KHZ800);

// 按键状态
bool button_press = false;

/**
* Function       keyscan
* @author        chengengyue
* @date          2019.10.08
* @brief         按键扫描
* @param[in1]    void
* @retval        void
* @par History   无
*/
void keyscan()
{
  int val;
  val = digitalRead(KEY_PIN); // 读取数字8口电平值赋给val
  if (val == LOW)             // 当按键被按下时
  {
    delay(10);                  // 延时消抖
    val = digitalRead(KEY_PIN); // 再次读取按键状态
    while (val == LOW)
    {
      val = digitalRead(KEY_PIN); // 第三次读取按键状态
      if (val == HIGH)            // 判断按键是否松开，松开才执行
      {
        button_press = !button_press;
        return;
      }
    }
  }
}

/*
* Function       whistle
* @author        chengengyue
* @date          2019.10.08
* @brief         鸣笛
* @param[in]     void
* @param[out]    void
* @retval        void
* @par History   无
*/
void whistle()
{
  for (int i = 0; i < 100; i++)
  {
    digitalWrite(BUZZER, HIGH); //发声音
    delay(3);                   //延时
    digitalWrite(BUZZER, LOW);  //不发声音
    delay(1);                   //延时
  }
}

/**
* Function       clearRGB
* @author        chengengyue
* @date          2019.10.08
* @brief         设置板载RGB灯灭
* @param[in1]   void
* @param[out]    void
* @retval        void
* @par History   无
*/
void clearRGB()
{
  uint32_t color = strip.Color(0, 0, 0);
  for (uint8_t i = 0; i < MAX_RGB; i++)
  {
    strip.setPixelColor(i, color);
  }
  strip.show();
}

/**
* Function       showRGB
* @author        chengengyue
* @date          2019.10.08
* @brief         控制RGB灯
* @param[in1]    num  选择哪个灯，如果大于等于最大值，则全亮
* @param[in2]    R    R值(0~255)
* @param[in3]    G    G值(0~255)
* @param[in4]    B    B值(0~255)
* @param[out]    void
* @retval        void
* @par History   无
*/
void showRGB(int num, int R, int G, int B)
{
  uint32_t color = strip.Color(G, R, B);
  if (num >= MAX_RGB)    // 全部一起显示
  {
    for (int i = 0; i < MAX_RGB; i++)
    {
      strip.setPixelColor(i, color);
    }
  }
  else                   // 单独显示某个灯
  {
    strip.setPixelColor(num, color);
  }
  strip.show();
}

/**
* Function       setup
* @author        chengengyue
* @date          2019.10.08
* @brief         初始化设置
* @param[in]     void
* @param[out]    void
* @retval        void
* @par History   无
*/
void setup() {
  // put your setup code here, to run once:
  pinMode(KEY_PIN, INPUT_PULLUP);// 设置功能按键引脚为输入模式
  pinMode(LED_PIN, OUTPUT);      // 设置LED引脚为输出模式
  pinMode(BUZZER, OUTPUT);       // 设置蜂鸣器引脚为输出模式
  pinMode(RGB_PIN, OUTPUT);      // 设置RGB灯引脚为输出模式
  
  strip.begin();
  strip.show();
  clearRGB();                      // 清除RGB灯
}

/**
* Function       loop
* @author        chengengyue
* @date          2019.10.08
* @brief         主循环函数
* @param[in]     void
* @param[out]    void
* @retval        void
* @par History   无
*/
void loop() {
  // put your main code here, to run repeatedly:
  // 按键扫描
  keyscan();

  if (button_press)
  {
    whistle();                       // 鸣笛
    showRGB(MAX_RGB, 255, 0, 0);     // 全部亮红灯0.5秒
    delay(500);
    showRGB(MAX_RGB, 0, 255, 0);     // 全部亮绿灯0.5秒
    delay(500);
    showRGB(MAX_RGB, 0, 0, 255);     // 全部亮蓝灯0.5秒
    delay(500);
    showRGB(MAX_RGB, 255, 255, 255); // 全部亮白灯0.5秒
    delay(500);
    button_press = false;            // 按键状态设为false
  }
  else 
  {
    clearRGB();                      // 清除RGB灯
  }
}
